# Frontend Demo - TypeScript Types from .NET API

This Vue 3 application demonstrates end-to-end type safety by consuming TypeScript types automatically generated from .NET API models.

## What This Demo Shows

This frontend showcases **automatic type generation and full type safety** between a .NET backend and Vue 3 frontend:

- ✅ **No manual type definitions needed** - Types are auto-generated from C# models
- ✅ **Full IntelliSense/autocomplete** - All properties, enums, and nested objects
- ✅ **Compile-time type checking** - Catches breaking changes before runtime
- ✅ **Vue template type safety** - Even template expressions are type-checked
- ✅ **Zero-config workflow** - No attributes or decorators required on C# models

## Architecture

```
┌─────────────────┐
│  .NET API       │
│  C# Models      │  Plain POCOs, no attributes needed
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│  Swagger/       │
│  OpenAPI Spec   │  Auto-generated by ASP.NET Core
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│  NSwag CLI      │  Converts OpenAPI → TypeScript
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│  NPM Package    │
│  @mycompany/    │  Published to GitLab Package Registry
│  api-types      │
└────────┬────────┘
         │
         ▼
┌─────────────────┐
│  Vue 3 Frontend │
│  (This App)     │  Consumes types with full type safety
└─────────────────┘
```

## Type Demonstrations

This app demonstrates all major TypeScript/C# type mapping scenarios:

### 1. **Complex Nested Objects**
- `Product` with nested `ProductDimensions`
- Deep object hierarchies maintained perfectly

### 2. **Enums**
- `OrderStatus`: Pending, Processing, Shipped, Delivered, Cancelled
- `PaymentStatus`: Pending, Authorized, Captured, Failed, Refunded
- `NotificationPriority`: Low, Normal, High, Urgent
- `CardType`, `BankAccountType`, `CryptoCurrency`, `OrganizationType`, `EmployeeRole`

### 3. **Polymorphic Types** (Inheritance)
- `Notification` base class with 4 implementations:
  - `EmailNotification` - with attachments
  - `SmsNotification` - with carrier info
  - `PushNotification` - with device tokens
  - `InAppNotification` - with action payloads

### 4. **Discriminated Unions**
- `PaymentMethod` with 4 implementations:
  - `CreditCardPayment`
  - `PayPalPayment`
  - `BankAccountPayment`
  - `CryptoPayment`

### 5. **Generic Types**
- `PaginatedResponse<T>` - Generic pagination wrapper
- `ApiResponse<T>` - Generic API response wrapper
- Full type inference for wrapped types

### 6. **Deep Nesting**
- `Organization` → `Department` → `Employee`
- `SecuritySettings`, `DepartmentBudget`, `ContactInfo`
- Nested object hierarchies preserved perfectly

### 7. **Date Mapping**
- C# `DateTime` → TypeScript `Date`
- Nullable dates handled correctly

### 8. **Nullable Types**
- C# nullable types (`T?`) → TypeScript union types (`T | null`)
- Proper null handling throughout

## Running the Demo

### Prerequisites
- Node.js 20+
- .NET 8.0 SDK (to generate types)

### Local Development

1. **Generate types from API** (one-time setup):
```bash
# From project root
cd src/Api/ProductApi
dotnet build
dotnet run &  # Start API in background
sleep 5

# Generate TypeScript types
curl http://localhost:5000/swagger/v1/swagger.json -o swagger.json
npx nswag openapi2tsclient \
  /input:swagger.json \
  /output:../../TypesPackage/src/models.ts \
  /template:fetch \
  /generateClientInterfaces:true \
  /generateClientClasses:false

# Build types package
cd ../../TypesPackage
npm install
npm run build

# Stop API
pkill -f ProductApi
```

2. **Run the frontend**:
```bash
cd ../Frontend
npm install
npm run dev
```

3. **View the app**: Open http://localhost:5173/

### Type Safety Verification

To verify full type checking including Vue templates:

```bash
npm run build  # Runs vue-tsc && vite build
```

This will:
- ✅ Type-check all TypeScript code
- ✅ Type-check all Vue template expressions
- ✅ Catch any mismatches between code and generated types
- ✅ Validate property access, method calls, and type conversions

## CI/CD Pipeline

The types are automatically generated and published on every commit:

1. **build-api**: Build .NET API, generate OpenAPI spec
2. **generate-types**: Run NSwag to generate TypeScript
3. **publish-npm-package**: Publish to GitLab Package Registry
4. **build-frontend**: Build this frontend using published package
5. **deploy-wiki**: Update wiki with generated types documentation

## Package Installation

In production, install the types package from the registry:

```bash
npm install @mycompany/api-types
```

For GitLab Package Registry, create `.npmrc`:
```
@mycompany:registry=https://gitlab.example.com/api/v4/packages/npm/
//gitlab.example.com/api/v4/packages/npm/:_authToken=${YOUR_TOKEN}
```

## Key Files

- `src/App.vue` - Main demo showcasing all type patterns
- `package.json` - Depends on `@mycompany/api-types`
- `tsconfig.json` - TypeScript config with strict checking
- `vite.config.ts` - Vue 3 + Vite configuration

## Type Safety Benefits

### Before (Manual Types)
```typescript
// ❌ Manually written, can drift from backend
interface Product {
  id: number;
  name: string;
  // ... prone to typos and outdated definitions
}
```

### After (Auto-Generated)
```typescript
// ✅ Auto-generated from C# models, always in sync
import type { Product } from '@mycompany/api-types';

const product: Product = {
  id: 1,
  name: 'Laptop',
  // ✅ IntelliSense for all properties
  // ✅ Compile error if API changes
  // ✅ Nested types work perfectly
  dimensions: {
    width: 35.6,
    height: 2.3,
    // ✅ Deep type safety
  }
};
```

## When API Models Change

1. Developer updates C# model in `src/Api/ProductApi/Models/`
2. Commits and pushes to Git
3. CI/CD automatically:
   - Regenerates OpenAPI spec
   - Regenerates TypeScript types
   - Publishes new package version
   - Updates wiki documentation
4. Frontend developers:
   - Run `npm install @mycompany/api-types@latest`
   - TypeScript immediately shows breaking changes
   - Fix any type errors
   - Deploy with confidence

## Technology Stack

- **Frontend**: Vue 3 + TypeScript + Vite
- **Backend**: .NET 8.0 + ASP.NET Core
- **API Spec**: OpenAPI/Swagger 3.0
- **Type Generation**: NSwag CLI
- **Package Registry**: GitLab Package Registry
- **CI/CD**: GitLab CI/CD

## Success Metrics

This POC successfully demonstrates:

✅ **Zero-effort type generation** - No attributes needed on C# models
✅ **Full type coverage** - 40+ types including complex patterns
✅ **Template type safety** - Vue templates are fully type-checked
✅ **Automated pipeline** - Types published automatically
✅ **Industry standards** - Uses OpenAPI/Swagger specification
✅ **Developer experience** - Full IntelliSense and compile-time safety

## Learn More

- [Main README](../../README.md) - Project overview and setup
- [API Models](../Api/ProductApi/Models/) - C# source models
- [Generated Types Package](../TypesPackage/) - Auto-generated TypeScript
- [GitLab CI/CD](.gitlab-ci.yml) - Automation pipeline
- [Wiki](../../wiki) - Full type reference documentation

---

**Last Updated**: 2025-10-31
**Vue Version**: 3.4
**TypeScript Version**: 5.0
**Type Generation**: NSwag 14.6.1
