namespace ProductApi.Models;

/// <summary>
/// Base payment method - demonstrates discriminated unions
/// </summary>
public abstract class PaymentMethod
{
    public int Id { get; set; }
    public bool IsDefault { get; set; }
    public DateTime CreatedAt { get; set; }
    public DateTime? LastUsedAt { get; set; }

    /// <summary>
    /// Payment method type discriminator
    /// </summary>
    public abstract string MethodType { get; }
}

/// <summary>
/// Credit card payment method
/// </summary>
public class CreditCardPayment : PaymentMethod
{
    public override string MethodType => "credit_card";
    public string CardholderName { get; set; } = string.Empty;
    public string Last4Digits { get; set; } = string.Empty;
    public string Brand { get; set; } = string.Empty; // Visa, Mastercard, etc.
    public int ExpirationMonth { get; set; }
    public int ExpirationYear { get; set; }
    public string? BillingZipCode { get; set; }
    public CardType CardType { get; set; }
}

/// <summary>
/// PayPal payment method
/// </summary>
public class PayPalPayment : PaymentMethod
{
    public override string MethodType => "paypal";
    public string Email { get; set; } = string.Empty;
    public string? PayPalAccountId { get; set; }
    public bool IsBusinessAccount { get; set; }
}

/// <summary>
/// Bank account payment method (ACH/Direct Debit)
/// </summary>
public class BankAccountPayment : PaymentMethod
{
    public override string MethodType => "bank_account";
    public string AccountHolderName { get; set; } = string.Empty;
    public string BankName { get; set; } = string.Empty;
    public string Last4Digits { get; set; } = string.Empty;
    public string RoutingNumber { get; set; } = string.Empty;
    public BankAccountType AccountType { get; set; }
}

/// <summary>
/// Cryptocurrency payment method
/// </summary>
public class CryptoPayment : PaymentMethod
{
    public override string MethodType => "crypto";
    public string WalletAddress { get; set; } = string.Empty;
    public CryptoCurrency Currency { get; set; }
    public string? Network { get; set; } // Ethereum Mainnet, Polygon, etc.
}

public enum CardType
{
    Credit,
    Debit,
    Prepaid
}

public enum BankAccountType
{
    Checking,
    Savings
}

public enum CryptoCurrency
{
    Bitcoin,
    Ethereum,
    USDC,
    USDT
}
