namespace ProductApi.Models;

/// <summary>
/// Base notification type - demonstrates polymorphism
/// </summary>
public abstract class Notification
{
    public int Id { get; set; }
    public string Title { get; set; } = string.Empty;
    public string Message { get; set; } = string.Empty;
    public DateTime CreatedAt { get; set; }
    public bool IsRead { get; set; }
    public NotificationPriority Priority { get; set; }

    /// <summary>
    /// Discriminator field for polymorphic deserialization
    /// </summary>
    public abstract string Type { get; }
}

/// <summary>
/// Email notification with email-specific properties
/// </summary>
public class EmailNotification : Notification
{
    public override string Type => "email";
    public string From { get; set; } = string.Empty;
    public string To { get; set; } = string.Empty;
    public string? ReplyTo { get; set; }
    public List<string> Cc { get; set; } = new();
    public List<EmailAttachment> Attachments { get; set; } = new();
}

/// <summary>
/// SMS notification with carrier information
/// </summary>
public class SmsNotification : Notification
{
    public override string Type => "sms";
    public string PhoneNumber { get; set; } = string.Empty;
    public string? Carrier { get; set; }
    public int CharacterCount { get; set; }
}

/// <summary>
/// Push notification for mobile devices
/// </summary>
public class PushNotification : Notification
{
    public override string Type => "push";
    public string DeviceToken { get; set; } = string.Empty;
    public string Platform { get; set; } = string.Empty; // iOS, Android, Web
    public Dictionary<string, string> CustomData { get; set; } = new();
    public string? ImageUrl { get; set; }
    public string? ActionUrl { get; set; }
}

/// <summary>
/// In-app notification shown in the application UI
/// </summary>
public class InAppNotification : Notification
{
    public override string Type => "in_app";
    public string ActionType { get; set; } = string.Empty; // navigate, modal, toast
    public string? ActionPayload { get; set; }
    public DateTime? ExpiresAt { get; set; }
}

public class EmailAttachment
{
    public string FileName { get; set; } = string.Empty;
    public string ContentType { get; set; } = string.Empty;
    public long SizeInBytes { get; set; }
    public string? Url { get; set; }
}

public enum NotificationPriority
{
    Low = 0,
    Normal = 1,
    High = 2,
    Urgent = 3
}
