using Microsoft.AspNetCore.Mvc;
using ProductApi.Models;

namespace ProductApi.Controllers;

[ApiController]
[Route("api/[controller]")]
public class ProductsController : ControllerBase
{
    private static readonly List<Product> _products = new()
    {
        new Product
        {
            Id = 1,
            Name = "Laptop Pro",
            Description = "High-performance laptop",
            Price = 1299.99m,
            StockQuantity = 50,
            Category = "Electronics",
            SKU = "LAP-001",
            CreatedAt = DateTime.UtcNow,
            IsActive = true,
            Tags = new List<string> { "electronics", "computers", "portable" },
            Dimensions = new ProductDimensions { Width = 35, Height = 2.5, Depth = 25, Weight = 2.1 }
        },
        new Product
        {
            Id = 2,
            Name = "Wireless Mouse",
            Description = "Ergonomic wireless mouse",
            Price = 29.99m,
            StockQuantity = 150,
            Category = "Accessories",
            SKU = "MOU-001",
            CreatedAt = DateTime.UtcNow,
            IsActive = true,
            Tags = new List<string> { "accessories", "peripherals" }
        }
    };

    [HttpGet]
    public ActionResult<IEnumerable<Product>> GetProducts()
    {
        return Ok(_products);
    }

    [HttpGet("{id}")]
    public ActionResult<Product> GetProduct(int id)
    {
        var product = _products.FirstOrDefault(p => p.Id == id);
        if (product == null)
            return NotFound();
        return Ok(product);
    }

    [HttpPost]
    public ActionResult<Product> CreateProduct(Product product)
    {
        product.Id = _products.Max(p => p.Id) + 1;
        product.CreatedAt = DateTime.UtcNow;
        _products.Add(product);
        return CreatedAtAction(nameof(GetProduct), new { id = product.Id }, product);
    }

    [HttpPut("{id}")]
    public IActionResult UpdateProduct(int id, Product product)
    {
        var existingProduct = _products.FirstOrDefault(p => p.Id == id);
        if (existingProduct == null)
            return NotFound();

        product.Id = id;
        product.UpdatedAt = DateTime.UtcNow;
        _products.Remove(existingProduct);
        _products.Add(product);
        return NoContent();
    }

    [HttpDelete("{id}")]
    public IActionResult DeleteProduct(int id)
    {
        var product = _products.FirstOrDefault(p => p.Id == id);
        if (product == null)
            return NotFound();

        _products.Remove(product);
        return NoContent();
    }
}