using Microsoft.AspNetCore.Mvc;
using ProductApi.Models;

namespace ProductApi.Controllers;

[ApiController]
[Route("api/[controller]")]
public class PaymentsController : ControllerBase
{
    /// <summary>
    /// Get all payment methods (demonstrates discriminated unions)
    /// </summary>
    [HttpGet("methods")]
    public ActionResult<List<PaymentMethod>> GetPaymentMethods()
    {
        var methods = new List<PaymentMethod>
        {
            new CreditCardPayment
            {
                Id = 1,
                IsDefault = true,
                CreatedAt = DateTime.UtcNow.AddMonths(-6),
                LastUsedAt = DateTime.UtcNow.AddDays(-2),
                CardholderName = "John Doe",
                Last4Digits = "4242",
                Brand = "Visa",
                ExpirationMonth = 12,
                ExpirationYear = 2025,
                BillingZipCode = "12345",
                CardType = CardType.Credit
            },
            new PayPalPayment
            {
                Id = 2,
                IsDefault = false,
                CreatedAt = DateTime.UtcNow.AddMonths(-3),
                LastUsedAt = DateTime.UtcNow.AddDays(-10),
                Email = "john.doe@example.com",
                PayPalAccountId = "PAYPAL123456",
                IsBusinessAccount = false
            },
            new BankAccountPayment
            {
                Id = 3,
                IsDefault = false,
                CreatedAt = DateTime.UtcNow.AddMonths(-12),
                LastUsedAt = DateTime.UtcNow.AddDays(-30),
                AccountHolderName = "John Doe",
                BankName = "Example Bank",
                Last4Digits = "9876",
                RoutingNumber = "021000021",
                AccountType = BankAccountType.Checking
            },
            new CryptoPayment
            {
                Id = 4,
                IsDefault = false,
                CreatedAt = DateTime.UtcNow.AddMonths(-1),
                LastUsedAt = null,
                WalletAddress = "0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb",
                Currency = CryptoCurrency.Ethereum,
                Network = "Ethereum Mainnet"
            }
        };

        return Ok(methods);
    }

    /// <summary>
    /// Get a specific credit card payment method
    /// </summary>
    [HttpGet("methods/credit-card/{id}")]
    public ActionResult<CreditCardPayment> GetCreditCard(int id)
    {
        var card = new CreditCardPayment
        {
            Id = id,
            IsDefault = true,
            CreatedAt = DateTime.UtcNow.AddYears(-1),
            LastUsedAt = DateTime.UtcNow,
            CardholderName = "Jane Smith",
            Last4Digits = "1234",
            Brand = "Mastercard",
            ExpirationMonth = 6,
            ExpirationYear = 2026,
            BillingZipCode = "54321",
            CardType = CardType.Debit
        };

        return Ok(card);
    }

    /// <summary>
    /// Get a specific PayPal payment method
    /// </summary>
    [HttpGet("methods/paypal/{id}")]
    public ActionResult<PayPalPayment> GetPayPal(int id)
    {
        var paypal = new PayPalPayment
        {
            Id = id,
            IsDefault = false,
            CreatedAt = DateTime.UtcNow.AddMonths(-6),
            LastUsedAt = DateTime.UtcNow.AddDays(-5),
            Email = "user@example.com",
            PayPalAccountId = "PAYPAL123456789",
            IsBusinessAccount = false
        };

        return Ok(paypal);
    }

    /// <summary>
    /// Get a specific bank account payment method
    /// </summary>
    [HttpGet("methods/bank/{id}")]
    public ActionResult<BankAccountPayment> GetBankAccount(int id)
    {
        var bank = new BankAccountPayment
        {
            Id = id,
            IsDefault = false,
            CreatedAt = DateTime.UtcNow.AddYears(-2),
            LastUsedAt = DateTime.UtcNow.AddMonths(-1),
            AccountHolderName = "John Smith",
            BankName = "Example National Bank",
            Last4Digits = "9876",
            RoutingNumber = "021000021",
            AccountType = BankAccountType.Checking
        };

        return Ok(bank);
    }

    /// <summary>
    /// Get a specific cryptocurrency payment method
    /// </summary>
    [HttpGet("methods/crypto/{id}")]
    public ActionResult<CryptoPayment> GetCrypto(int id)
    {
        var crypto = new CryptoPayment
        {
            Id = id,
            IsDefault = false,
            CreatedAt = DateTime.UtcNow.AddMonths(-3),
            LastUsedAt = null,
            WalletAddress = "0x742d35Cc6634C0532925a3b844Bc9e7595f0bEb",
            Currency = CryptoCurrency.Ethereum,
            Network = "Ethereum Mainnet"
        };

        return Ok(crypto);
    }

    /// <summary>
    /// Get payment methods wrapped in API response
    /// </summary>
    [HttpGet("methods/wrapped")]
    public ActionResult<ApiResponse<List<PaymentMethod>>> GetPaymentMethodsWrapped()
    {
        var methods = new List<PaymentMethod>
        {
            new CreditCardPayment
            {
                Id = 1,
                IsDefault = true,
                CreatedAt = DateTime.UtcNow,
                CardholderName = "Test User",
                Last4Digits = "4242",
                Brand = "Visa",
                ExpirationMonth = 12,
                ExpirationYear = 2025,
                CardType = CardType.Credit
            }
        };

        var response = new ApiResponse<List<PaymentMethod>>
        {
            Success = true,
            Data = methods,
            Error = null,
            ValidationErrors = null,
            Timestamp = DateTime.UtcNow
        };

        return Ok(response);
    }

    /// <summary>
    /// Simulate an error response
    /// </summary>
    [HttpGet("methods/error")]
    public ActionResult<ApiResponse<List<PaymentMethod>>> GetPaymentMethodsError()
    {
        var response = new ApiResponse<List<PaymentMethod>>
        {
            Success = false,
            Data = null,
            Error = new ApiError
            {
                Code = "PAYMENT_METHOD_NOT_FOUND",
                Message = "No payment methods found for this user",
                Details = "User has not added any payment methods yet"
            },
            ValidationErrors = new Dictionary<string, string>
            {
                { "userId", "User ID is required" }
            },
            Timestamp = DateTime.UtcNow
        };

        return NotFound(response);
    }
}
