using Microsoft.AspNetCore.Mvc;
using ProductApi.Models;

namespace ProductApi.Controllers;

[ApiController]
[Route("api/[controller]")]
public class OrganizationsController : ControllerBase
{
    /// <summary>
    /// Get an organization with complex nested structure
    /// </summary>
    [HttpGet("{id}")]
    public ActionResult<Organization> GetOrganization(int id)
    {
        var organization = new Organization
        {
            Id = id,
            Name = "Acme Corporation",
            Description = "A leading technology company",
            Type = OrganizationType.Corporation,
            FoundedDate = new DateTime(2010, 1, 15),
            Settings = new OrganizationSettings
            {
                AllowPublicProfile = true,
                MaxEmployees = 500,
                TimeZone = TimeZoneInfo.Local,
                Currency = "USD",
                AllowedDomains = new List<string> { "acme.com", "acmecorp.com" },
                Security = new SecuritySettings
                {
                    RequireTwoFactor = true,
                    RequireStrongPasswords = true,
                    PasswordExpirationDays = 90,
                    MaxLoginAttempts = 5,
                    AllowedIpRanges = new List<string> { "192.168.1.0/24", "10.0.0.0/8" }
                }
            },
            Departments = new List<Department>
            {
                new Department
                {
                    Id = 1,
                    Name = "Engineering",
                    Code = "ENG",
                    ParentDepartmentId = null,
                    Manager = new Employee
                    {
                        Id = 101,
                        FirstName = "Alice",
                        LastName = "Johnson",
                        Email = "alice.johnson@acme.com",
                        PhoneNumber = "+1-555-0101",
                        Role = EmployeeRole.Director,
                        HireDate = new DateTime(2015, 3, 1),
                        Salary = 150000m,
                        HomeAddress = new Address
                        {
                            Street = "123 Tech Street",
                            City = "San Francisco",
                            State = "CA",
                            PostalCode = "94102",
                            Country = "USA",
                            Latitude = 37.7749,
                            Longitude = -122.4194
                        },
                        Skills = new List<string> { "Leadership", "Software Architecture", "Cloud Computing" },
                        Certifications = new Dictionary<string, string>
                        {
                            { "AWS Solutions Architect", "2023-05-15" },
                            { "PMP", "2022-08-20" }
                        }
                    },
                    Employees = new List<Employee>
                    {
                        new Employee
                        {
                            Id = 201,
                            FirstName = "Bob",
                            LastName = "Smith",
                            Email = "bob.smith@acme.com",
                            Role = EmployeeRole.Employee,
                            HireDate = new DateTime(2020, 6, 15),
                            Salary = 95000m,
                            Skills = new List<string> { "C#", "TypeScript", "React" },
                            Certifications = new Dictionary<string, string>()
                        },
                        new Employee
                        {
                            Id = 202,
                            FirstName = "Carol",
                            LastName = "Williams",
                            Email = "carol.williams@acme.com",
                            Role = EmployeeRole.Manager,
                            HireDate = new DateTime(2018, 9, 1),
                            Salary = 115000m,
                            Skills = new List<string> { "Python", "Machine Learning", "DevOps" },
                            Certifications = new Dictionary<string, string>
                            {
                                { "Kubernetes Administrator", "2023-01-10" }
                            }
                        }
                    },
                    Budget = new DepartmentBudget
                    {
                        AnnualBudget = 5000000m,
                        SpentToDate = 3200000m,
                        RemainingBudget = 1800000m,
                        FiscalYear = 2024,
                        CategoryBreakdown = new Dictionary<string, decimal>
                        {
                            { "Salaries", 3000000m },
                            { "Equipment", 500000m },
                            { "Training", 200000m },
                            { "Software Licenses", 300000m }
                        }
                    },
                    Responsibilities = new List<string>
                    {
                        "Product Development",
                        "Infrastructure Management",
                        "Technical Support"
                    }
                },
                new Department
                {
                    Id = 2,
                    Name = "Marketing",
                    Code = "MKT",
                    ParentDepartmentId = null,
                    Manager = new Employee
                    {
                        Id = 102,
                        FirstName = "David",
                        LastName = "Brown",
                        Email = "david.brown@acme.com",
                        Role = EmployeeRole.VP,
                        HireDate = new DateTime(2012, 7, 1),
                        Salary = 175000m,
                        Skills = new List<string> { "Marketing Strategy", "Brand Management", "Digital Marketing" },
                        Certifications = new Dictionary<string, string>()
                    },
                    Employees = new List<Employee>(),
                    Budget = new DepartmentBudget
                    {
                        AnnualBudget = 2000000m,
                        SpentToDate = 1500000m,
                        RemainingBudget = 500000m,
                        FiscalYear = 2024,
                        CategoryBreakdown = new Dictionary<string, decimal>
                        {
                            { "Advertising", 1000000m },
                            { "Events", 300000m },
                            { "Content Creation", 200000m }
                        }
                    },
                    Responsibilities = new List<string>
                    {
                        "Brand Development",
                        "Customer Acquisition",
                        "Market Research"
                    }
                }
            },
            CustomFields = new Dictionary<string, string>
            {
                { "industry", "Technology" },
                { "stockSymbol", "ACME" },
                { "numberOfOffices", "5" }
            },
            HeadquartersAddress = new Address
            {
                Street = "1 Innovation Way",
                City = "San Francisco",
                State = "CA",
                PostalCode = "94105",
                Country = "USA",
                Latitude = 37.7899,
                Longitude = -122.3966
            },
            ContactInfo = new ContactInfo
            {
                PrimaryPhone = "+1-800-555-ACME",
                SecondaryPhone = "+1-800-555-HELP",
                Email = "info@acme.com",
                Website = "https://www.acme.com",
                SocialMedia = new Dictionary<string, string>
                {
                    { "Twitter", "@acmecorp" },
                    { "LinkedIn", "acme-corporation" },
                    { "GitHub", "acmecorp" }
                }
            }
        };

        return Ok(organization);
    }

    /// <summary>
    /// Get all organizations with pagination
    /// </summary>
    [HttpGet]
    public ActionResult<PaginatedResponse<Organization>> GetOrganizations(
        [FromQuery] int page = 1,
        [FromQuery] int pageSize = 10)
    {
        // Simplified organization for list view
        var organizations = new List<Organization>
        {
            new Organization
            {
                Id = 1,
                Name = "Acme Corporation",
                Type = OrganizationType.Corporation,
                FoundedDate = new DateTime(2010, 1, 15),
                Departments = new List<Department>(),
                CustomFields = new Dictionary<string, string>()
            },
            new Organization
            {
                Id = 2,
                Name = "TechStart LLC",
                Type = OrganizationType.LLC,
                FoundedDate = new DateTime(2018, 5, 20),
                Departments = new List<Department>(),
                CustomFields = new Dictionary<string, string>()
            }
        };

        var totalItems = organizations.Count;
        var totalPages = (int)Math.Ceiling(totalItems / (double)pageSize);

        var response = new PaginatedResponse<Organization>
        {
            Items = organizations,
            Pagination = new PaginationMetadata
            {
                CurrentPage = page,
                PageSize = pageSize,
                TotalItems = totalItems,
                TotalPages = totalPages,
                HasNextPage = page < totalPages,
                HasPreviousPage = page > 1
            }
        };

        return Ok(response);
    }
}
