using Microsoft.AspNetCore.Mvc;
using ProductApi.Models;

namespace ProductApi.Controllers;

[ApiController]
[Route("api/[controller]")]
public class NotificationsController : ControllerBase
{
    /// <summary>
    /// Get all notifications (demonstrates polymorphic type handling)
    /// </summary>
    [HttpGet]
    public ActionResult<List<Notification>> GetNotifications()
    {
        var notifications = new List<Notification>
        {
            new EmailNotification
            {
                Id = 1,
                Title = "Welcome Email",
                Message = "Welcome to our platform!",
                CreatedAt = DateTime.UtcNow.AddHours(-2),
                IsRead = true,
                Priority = NotificationPriority.Normal,
                From = "noreply@example.com",
                To = "user@example.com",
                Cc = new List<string> { "admin@example.com" },
                Attachments = new List<EmailAttachment>
                {
                    new EmailAttachment
                    {
                        FileName = "welcome.pdf",
                        ContentType = "application/pdf",
                        SizeInBytes = 1024000
                    }
                }
            },
            new SmsNotification
            {
                Id = 2,
                Title = "Verification Code",
                Message = "Your code is 123456",
                CreatedAt = DateTime.UtcNow.AddMinutes(-30),
                IsRead = false,
                Priority = NotificationPriority.High,
                PhoneNumber = "+1234567890",
                Carrier = "Verizon",
                CharacterCount = 24
            },
            new PushNotification
            {
                Id = 3,
                Title = "New Message",
                Message = "You have a new message",
                CreatedAt = DateTime.UtcNow.AddMinutes(-5),
                IsRead = false,
                Priority = NotificationPriority.Urgent,
                DeviceToken = "abc123xyz789",
                Platform = "iOS",
                ImageUrl = "https://example.com/image.png",
                ActionUrl = "/messages/123",
                CustomData = new Dictionary<string, string>
                {
                    { "senderId", "user456" },
                    { "messageType", "direct" }
                }
            },
            new InAppNotification
            {
                Id = 4,
                Title = "System Update",
                Message = "A new version is available",
                CreatedAt = DateTime.UtcNow.AddDays(-1),
                IsRead = true,
                Priority = NotificationPriority.Low,
                ActionType = "modal",
                ActionPayload = "{\"version\": \"2.0.0\"}",
                ExpiresAt = DateTime.UtcNow.AddDays(7)
            }
        };

        return Ok(notifications);
    }

    /// <summary>
    /// Get a specific email notification
    /// </summary>
    [HttpGet("email/{id}")]
    public ActionResult<EmailNotification> GetEmailNotification(int id)
    {
        var notification = new EmailNotification
        {
            Id = id,
            Title = "Order Confirmation",
            Message = "Your order has been confirmed",
            CreatedAt = DateTime.UtcNow,
            IsRead = false,
            Priority = NotificationPriority.High,
            From = "orders@example.com",
            To = "customer@example.com",
            ReplyTo = "support@example.com",
            Attachments = new List<EmailAttachment>()
        };

        return Ok(notification);
    }

    /// <summary>
    /// Get a specific SMS notification
    /// </summary>
    [HttpGet("sms/{id}")]
    public ActionResult<SmsNotification> GetSmsNotification(int id)
    {
        var notification = new SmsNotification
        {
            Id = id,
            Title = "Verification Code",
            Message = "Your verification code is 123456",
            CreatedAt = DateTime.UtcNow,
            IsRead = false,
            Priority = NotificationPriority.High,
            PhoneNumber = "+1234567890",
            Carrier = "Verizon",
            CharacterCount = 37
        };

        return Ok(notification);
    }

    /// <summary>
    /// Get a specific push notification
    /// </summary>
    [HttpGet("push/{id}")]
    public ActionResult<PushNotification> GetPushNotification(int id)
    {
        var notification = new PushNotification
        {
            Id = id,
            Title = "New Message",
            Message = "You have a new message from John",
            CreatedAt = DateTime.UtcNow,
            IsRead = false,
            Priority = NotificationPriority.Urgent,
            DeviceToken = "abc123xyz789",
            Platform = "iOS",
            ImageUrl = "https://example.com/avatar.png",
            ActionUrl = "/messages/123",
            CustomData = new Dictionary<string, string>
            {
                { "senderId", "user456" },
                { "messageType", "direct" }
            }
        };

        return Ok(notification);
    }

    /// <summary>
    /// Get a specific in-app notification
    /// </summary>
    [HttpGet("inapp/{id}")]
    public ActionResult<InAppNotification> GetInAppNotification(int id)
    {
        var notification = new InAppNotification
        {
            Id = id,
            Title = "System Update Available",
            Message = "Version 2.0.0 is now available",
            CreatedAt = DateTime.UtcNow,
            IsRead = false,
            Priority = NotificationPriority.Low,
            ActionType = "modal",
            ActionPayload = "{\"version\": \"2.0.0\", \"releaseNotes\": \"...\"}",
            ExpiresAt = DateTime.UtcNow.AddDays(7)
        };

        return Ok(notification);
    }

    /// <summary>
    /// Get paginated notifications
    /// </summary>
    [HttpGet("paginated")]
    public ActionResult<PaginatedResponse<Notification>> GetPaginatedNotifications(
        [FromQuery] int page = 1,
        [FromQuery] int pageSize = 10)
    {
        var allNotifications = new List<Notification>
        {
            new EmailNotification { Id = 1, Title = "Email 1", Message = "Message 1", CreatedAt = DateTime.UtcNow, Priority = NotificationPriority.Normal },
            new SmsNotification { Id = 2, Title = "SMS 1", Message = "Message 2", CreatedAt = DateTime.UtcNow, Priority = NotificationPriority.High, PhoneNumber = "+1234567890", CharacterCount = 20 },
            new PushNotification { Id = 3, Title = "Push 1", Message = "Message 3", CreatedAt = DateTime.UtcNow, Priority = NotificationPriority.Normal, DeviceToken = "token123", Platform = "Android" }
        };

        var totalItems = allNotifications.Count;
        var totalPages = (int)Math.Ceiling(totalItems / (double)pageSize);
        var items = allNotifications.Skip((page - 1) * pageSize).Take(pageSize).ToList();

        var response = new PaginatedResponse<Notification>
        {
            Items = items,
            Pagination = new PaginationMetadata
            {
                CurrentPage = page,
                PageSize = pageSize,
                TotalItems = totalItems,
                TotalPages = totalPages,
                HasNextPage = page < totalPages,
                HasPreviousPage = page > 1,
                NextPageUrl = page < totalPages ? $"/api/notifications/paginated?page={page + 1}&pageSize={pageSize}" : null,
                PreviousPageUrl = page > 1 ? $"/api/notifications/paginated?page={page - 1}&pageSize={pageSize}" : null
            },
            Metadata = new Dictionary<string, object>
            {
                { "filterApplied", false },
                { "sortBy", "createdAt" }
            }
        };

        return Ok(response);
    }
}
