# .NET to TypeScript Type Sharing POC

> **Automatically generate and distribute TypeScript types from .NET models via OpenAPI/Swagger**

This proof of concept demonstrates a complete workflow for sharing .NET API models with frontend applications through TypeScript packages. Changes to C# models are automatically converted to TypeScript types via OpenAPI specification and published as an npm package.

## 📋 Table of Contents

- [Overview](#overview)
- [Quick Start](#quick-start)
- [Implementation Guide](#implementation-guide)
- [Project Structure](#project-structure)
- [How It Works](#how-it-works)
- [Development Workflow](#development-workflow)
- [Troubleshooting](#troubleshooting)

---

## Overview

### The Problem
Frontend applications need type-safe access to backend API models, but manually maintaining TypeScript definitions is error-prone and time-consuming.

### The Solution
Automatically generate TypeScript types from C# models using OpenAPI/Swagger specification and NSwag during the CI/CD build process, then distribute them as an npm package.

### Key Benefits
- ✅ **Single source of truth** - C# models define the contract
- ✅ **Type safety** - Full IntelliSense in frontend code
- ✅ **Automatic updates** - Types regenerate on every build
- ✅ **Easy distribution** - Standard npm package workflow
- ✅ **No runtime overhead** - Compile-time only
- ✅ **OpenAPI standard** - Uses industry-standard Swagger/OpenAPI specification

---

## Quick Start

### Prerequisites
- .NET 8.0 SDK
- Node.js 20+
- GitLab account (for CI/CD)

### Run the Complete Build

The full build pipeline is automated in CI/CD. Locally, you can run individual components:

```bash
# 1. Run the API
cd src/Api/ProductApi
dotnet run

# API starts on http://localhost:5000
# Swagger UI available at http://localhost:5000/swagger
```

In another terminal:

```bash
# 2. Fetch OpenAPI spec and generate types
curl http://localhost:5000/swagger/v1/swagger.json -o swagger.json
npx nswag openapi2tsclient /input:swagger.json /output:src/TypesPackage/src/models.ts

# 3. Build the TypeScript package
cd src/TypesPackage
npm install
npm run build

# 4. Run the frontend demo
cd ../Frontend
npm install
npm run dev
```

Visit http://localhost:5173 to see the demo.

---

## Implementation Guide

Follow these steps to implement this pattern in your own project.

### Step 1: Install NSwag in Your API Project

Add the NuGet packages to your .NET project:

```xml
<!-- YourApi.csproj -->
<ItemGroup>
  <PackageReference Include="Swashbuckle.AspNetCore" Version="6.5.0" />
  <PackageReference Include="NSwag.AspNetCore" Version="14.6.1" />
</ItemGroup>
```

### Step 2: Configure Swagger/OpenAPI in Program.cs

Enable Swagger in **all environments** (not just Development):

```csharp
var builder = WebApplication.CreateBuilder(args);

builder.Services.AddControllers();
builder.Services.AddEndpointsApiExplorer();
builder.Services.AddSwaggerGen(c =>
{
    c.SwaggerDoc("v1", new Microsoft.OpenApi.Models.OpenApiInfo
    {
        Title = "Your API",
        Version = "v1",
        Description = "API with auto-generated TypeScript types"
    });
});

var app = builder.Build();

// Enable Swagger in ALL environments (including Production)
// This is required for CI/CD type generation
app.UseSwagger();
app.UseSwaggerUI(c =>
{
    c.SwaggerEndpoint("/swagger/v1/swagger.json", "Your API V1");
});

app.UseAuthorization();
app.MapControllers();

app.Run();
```

**Important:** Swagger must be enabled in Production for the CI/CD pipeline to fetch the OpenAPI spec during build.

### Step 3: Define Your Models

Create your C# models as usual - no special attributes required:

```csharp
namespace YourApi.Models;

/// <summary>
/// Represents a product in the catalog
/// </summary>
public class Product
{
    public int Id { get; set; }
    public string Name { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public DateTime CreatedAt { get; set; }
    public DateTime? UpdatedAt { get; set; }
    public List<string> Tags { get; set; } = new();
    public ProductStatus Status { get; set; }
}

public enum ProductStatus
{
    Active,
    Inactive,
    OutOfStock
}
```

### Step 4: Set Up Your TypeScript Package Structure

The TypeScript package is **generated during CI/CD** - it should NOT be checked into source control. Add to `.gitignore`:

```gitignore
# Generated TypeScript package (created during build/CI)
src/TypesPackage/
```

The CI pipeline will create this structure:

```
TypesPackage/
├── package.json          # Generated by CI
├── tsconfig.json         # Generated by CI
├── src/
│   ├── models.ts         # Generated by NSwag from OpenAPI spec
│   └── index.ts          # Generated by CI
└── dist/                 # Built by TypeScript compiler
    ├── index.js
    ├── index.d.ts
    ├── models.js
    └── models.d.ts
```

### Step 5: Set Up GitLab CI/CD

Create `.gitlab-ci.yml` in your repository root:

```yaml
stages:
  - build
  - generate-types
  - publish-package
  - deploy

variables:
  DOTNET_VERSION: "8.0"
  NODE_VERSION: "20"
  API_PROJECT_PATH: "src/Api/YourApi"
  TYPES_PACKAGE_PATH: "src/TypesPackage"

build-api:
  stage: build
  image: mcr.microsoft.com/dotnet/sdk:8.0
  before_script:
    - apt-get update && apt-get install -y nodejs npm
  script:
    # Build the API
    - cd $API_PROJECT_PATH
    - dotnet restore
    - dotnet build -c Release
    - dotnet publish -c Release -o ./publish --self-contained --runtime linux-x64

    # Return to root
    - cd ../../..
    - mkdir -p $TYPES_PACKAGE_PATH/src

    # Start API temporarily to fetch OpenAPI spec
    - cd $API_PROJECT_PATH
    - nohup dotnet bin/Release/net8.0/YourApi.dll > api.log 2>&1 &
    - API_PID=$!
    - sleep 10

    # Fetch OpenAPI spec
    - curl -f http://localhost:8080/swagger/v1/swagger.json -o swagger.json
    - kill $API_PID || true

    # Return to root
    - cd ../../..

    # Generate TypeScript from OpenAPI spec using NSwag
    - npm install -g nswag
    - nswag openapi2tsclient /input:$API_PROJECT_PATH/swagger.json /output:$TYPES_PACKAGE_PATH/src/models.ts /template:fetch /generateClientInterfaces:true /generateClientClasses:false /typeStyle:interface /enumStyle:enum

    # Create package.json
    - |
      cat > $TYPES_PACKAGE_PATH/package.json <<'PKGJSON'
      {
        "name": "@yourcompany/api-types",
        "version": "1.0.$CI_PIPELINE_IID",
        "description": "Auto-generated TypeScript types from .NET API",
        "main": "dist/index.js",
        "types": "dist/index.d.ts",
        "scripts": {
          "build": "tsc"
        },
        "devDependencies": {
          "typescript": "^5.0.0",
          "@types/node": "^20.0.0"
        },
        "files": ["dist/**/*", "src/**/*"]
      }
      PKGJSON

    # Create tsconfig.json
    - |
      cat > $TYPES_PACKAGE_PATH/tsconfig.json <<'TSCONFIG'
      {
        "compilerOptions": {
          "target": "ES2020",
          "module": "commonjs",
          "lib": ["ES2020", "DOM"],
          "declaration": true,
          "declarationMap": true,
          "outDir": "./dist",
          "rootDir": "./src",
          "strict": true,
          "esModuleInterop": true,
          "skipLibCheck": true
        },
        "include": ["src/**/*"],
        "exclude": ["node_modules", "dist"]
      }
      TSCONFIG

    # Create index.ts
    - |
      cat > $TYPES_PACKAGE_PATH/src/index.ts <<'INDEX'
      export * from './models';
      export const PACKAGE_VERSION = '1.0.$CI_PIPELINE_IID';
      INDEX

  artifacts:
    paths:
      - $API_PROJECT_PATH/publish/
      - $TYPES_PACKAGE_PATH/
    expire_in: 1 week

generate-types:
  stage: generate-types
  image: node:20-alpine
  dependencies:
    - build-api
  script:
    - cd $TYPES_PACKAGE_PATH
    - npm install
    - npm run build
  artifacts:
    paths:
      - $TYPES_PACKAGE_PATH/
    expire_in: 1 week

publish-npm-package:
  stage: publish-package
  image: node:20-alpine
  dependencies:
    - generate-types
  only:
    - main
    - master
    - tags
  script:
    - cd $TYPES_PACKAGE_PATH
    - echo "//gitlab/api/v4/projects/${CI_PROJECT_ID}/packages/npm/:_authToken=${CI_JOB_TOKEN}" > .npmrc
    - npm publish --registry="http://gitlab/api/v4/projects/${CI_PROJECT_ID}/packages/npm/"
```

**Key Points:**
- Uses internal Docker URL `http://gitlab/...` for package registry
- Starts API temporarily during build to fetch OpenAPI spec
- Generates types using NSwag CLI
- Auto-increments version using `$CI_PIPELINE_IID`

### Step 6: (Optional) Enable Wiki Documentation

To automatically deploy TypeScript types to your GitLab wiki:

1. **Create a Project Access Token:**
   - Go to **Settings → Access Tokens**
   - Name: `Wiki Deployment Token`
   - Scopes: Check `api`
   - Role: `Maintainer` or `Owner`
   - Copy the generated token

2. **Add as CI/CD Variable:**
   - Go to **Settings → CI/CD → Variables**
   - Key: `WIKI_TOKEN`
   - Value: Paste the token
   - Protect variable: ✅
   - Mask variable: ✅

The wiki page will be automatically created/updated on each pipeline run at:
`https://your-gitlab.com/your-namespace/your-project/-/wikis/TypeScript-Types`

**Why a Project Access Token?**
GitLab's `CI_JOB_TOKEN` doesn't support the Wiki API endpoint. The pipeline gracefully skips wiki deployment if `WIKI_TOKEN` is not set.

### Step 7: Use in Your Frontend

Configure npm to use GitLab Package Registry (`.npmrc` in your frontend project):

```
@yourcompany:registry=https://gitlab.yourcompany.com/api/v4/packages/npm/
//gitlab.yourcompany.com/api/v4/packages/npm/:_authToken=${CI_JOB_TOKEN}
```

Install and use the types:

```bash
npm install @yourcompany/api-types
```

```typescript
import type { Product, ProductStatus } from '@yourcompany/api-types';

// Full type safety and IntelliSense
const product: Product = {
  id: 1,
  name: 'Sample Product',
  price: 99.99,
  createdAt: new Date(),
  updatedAt: null,
  tags: ['electronics', 'new'],
  status: ProductStatus.Active
};
```

---

## Project Structure

```
type-package-poc/
├── src/
│   ├── Api/ProductApi/              # .NET 8.0 Web API
│   │   ├── Controllers/             # API endpoints
│   │   ├── Models/                  # C# models (source of truth)
│   │   │   ├── Product.cs
│   │   │   └── Order.cs
│   │   ├── ProductApi.csproj        # Includes NSwag.AspNetCore
│   │   └── Program.cs               # Swagger configuration
│   │
│   ├── TypesPackage/                # [GENERATED] - Not in source control
│   │   ├── src/
│   │   │   ├── models.ts            # Generated by NSwag from OpenAPI
│   │   │   └── index.ts             # Package exports
│   │   ├── package.json             # Created by CI
│   │   └── tsconfig.json            # Created by CI
│   │
│   └── Frontend/                    # React demo application
│       ├── src/
│       │   ├── App.tsx              # Demo using the types
│       │   └── main.tsx
│       ├── package.json             # Depends on @mycompany/api-types
│       └── vite.config.ts
│
├── .gitlab-ci.yml                   # CI/CD pipeline configuration
├── .gitignore                       # Ignores src/TypesPackage/
└── README.md                        # This file
```

---

## How It Works

### 1. Type Generation Flow

```
C# Models (Product.cs)
        ↓
.NET API Build
        ↓
OpenAPI/Swagger Specification (swagger.json)
        ↓
NSwag CLI (openapi2tsclient)
        ↓
TypeScript Interfaces (models.ts)
        ↓
TypeScript Compiler (tsc)
        ↓
NPM Package (dist/*.js + *.d.ts)
        ↓
Frontend Application
```

### 2. Example Transformation

**C# Model:**
```csharp
public class Product
{
    public int Id { get; set; }
    public string Name { get; set; }
    public decimal Price { get; set; }
    public DateTime CreatedAt { get; set; }
    public DateTime? UpdatedAt { get; set; }
    public List<string> Tags { get; set; }
    public ProductDimensions? Dimensions { get; set; }
}

public class ProductDimensions
{
    public double Width { get; set; }
    public double Height { get; set; }
    public string Unit { get; set; }
}
```

**Generated TypeScript:**
```typescript
export interface Product {
    id?: number;
    name?: string | null;
    price?: number;
    createdAt?: Date;
    updatedAt?: Date | null;
    tags?: string[] | null;
    dimensions?: ProductDimensions;
}

export interface ProductDimensions {
    width?: number;
    height?: number;
    unit?: string | null;
}
```

**Type Mappings:**
- `int`, `double`, `decimal` → `number`
- `string` → `string`
- `DateTime` → `Date`
- `bool` → `boolean`
- `List<T>` → `T[]`
- `T?` (nullable) → `T | null`
- Nested objects preserved

### 3. CI/CD Pipeline

The GitLab pipeline automates the entire process:

1. **build-api** - Build API, start temporarily, fetch OpenAPI spec, generate TypeScript with NSwag
2. **generate-types** - Install dependencies and compile TypeScript package
3. **publish-npm-package** - Publish to GitLab Package Registry (main/master/tags only)
4. **deploy** - Create API artifacts for deployment

---

## Development Workflow

### Making Changes

1. **Update C# models** in `src/Api/ProductApi/Models/`
2. **Add/update API endpoints** in Controllers
3. **Commit and push** - CI/CD regenerates types and publishes new version
4. **Update frontend** - `npm update @yourcompany/api-types`

### Local Development

For local frontend development with the types package:

1. Build the API and generate types locally (see Quick Start)
2. Use file reference in `package.json`:

```json
{
  "dependencies": {
    "@mycompany/api-types": "file:../TypesPackage"
  }
}
```

### Version Management

The package version auto-increments with each pipeline run using `CI_PIPELINE_IID`.

For semantic versioning, you can modify the `package.json` generation in `.gitlab-ci.yml`:

```json
"version": "1.2.$CI_PIPELINE_IID"
```

Or use Git tags for releases.

---

## Troubleshooting

### Types Not Generating

**Problem:** TypeScript files aren't created after building

**Solution:**
1. Verify Swagger is enabled in Program.cs for ALL environments
2. Check API starts successfully (check api.log in CI)
3. Ensure OpenAPI spec is accessible at `/swagger/v1/swagger.json`
4. Verify NSwag is installed in build-api job
5. Check CI job logs for NSwag errors

### Swagger 404 in CI

**Problem:** `curl: (22) The requested URL returned error: 404`

**Solution:**
- Ensure `app.UseSwagger()` is called in ALL environments, not just Development
- Check that API port matches curl command (8080 in Production, 5000 in Development)
- Increase sleep time if API takes longer to start

### Package Not Found in Frontend

**Problem:** Frontend can't find the types package

**Solution:**
1. Check `.npmrc` is configured correctly for GitLab registry
2. Ensure package was published (check publish-npm-package job)
3. Verify package name matches in both places
4. For local dev, ensure TypesPackage is built first

### CI/CD Pipeline Fails

**Problem:** Pipeline jobs fail

**Common Issues:**
- **No runners**: Remove restrictive `tags:` from jobs
- **Connection refused**: Use internal Docker URLs (`http://gitlab/...`)
- **API won't start**: Check for port conflicts or missing dependencies
- **NSwag errors**: Ensure OpenAPI spec is valid JSON

### Type Mismatches

**Problem:** TypeScript types don't match C# models

**Solution:**
1. Check OpenAPI spec is correct: `curl http://localhost:5000/swagger/v1/swagger.json`
2. Verify C# models have proper XML documentation for better OpenAPI generation
3. Rebuild and republish the types package
4. Clear frontend TypeScript cache: `rm -rf node_modules/.cache`

---

## GitHub Actions Alternative

This project includes both GitLab CI/CD and GitHub Actions workflows. To use GitHub Actions:

1. **Copy `.github/workflows/build-and-publish.yml`** to your repository
2. **Enable GitHub Packages** in your repository settings
3. **Enable GitHub Wiki** (Settings → Features → Wikis)
4. **Configure package scope** in the workflow to match your organization

The GitHub Actions workflow publishes to **GitHub Packages** instead of GitLab Package Registry and deploys to **GitHub Wiki** instead of GitLab Wiki.

### Key Differences:
- **GitLab**: Uses `CI_JOB_TOKEN` for package registry, requires `WIKI_TOKEN` for wiki
- **GitHub**: Uses `GITHUB_TOKEN` for both packages and wiki deployment
- **Package Registry**: GitLab Package Registry vs GitHub Packages
- **URLs**: Internal Docker URLs (`http://gitlab/...`) vs public URLs

Both workflows follow the same 6-stage pattern:
1. Build API and generate types
2. Compile TypeScript package
3. Publish to package registry
4. Deploy to wiki
5. Build frontend demo
6. Create release (tags only)

---

## Additional Resources

- [NSwag Documentation](https://github.com/RicoSuter/NSwag)
- [OpenAPI Specification](https://swagger.io/specification/)
- [GitLab Package Registry Docs](https://docs.gitlab.com/ee/user/packages/npm_registry/)
- [GitHub Packages Docs](https://docs.github.com/en/packages)
- [TypeScript Handbook](https://www.typescriptlang.org/docs/)

---

## License

MIT

---

**Questions or Issues?** Open an issue in this repository.
