# CLAUDE.md

This file provides guidance to Claude Code (claude.ai/code) when working with code in this repository.

## Project Overview

This is a TypeScript Package POC demonstrating automatic type generation from .NET C# models to TypeScript via OpenAPI/Swagger specification, distributed via npm. The architecture consists of:
- **Backend**: .NET 8.0 Web API (`/src/Api/ProductApi/`) with Swagger for OpenAPI generation
- **Types Package**: Auto-generated TypeScript npm package (`/src/TypesPackage/`) - **NOT in source control**
- **Frontend Demo**: React application (`/src/Frontend/`) consuming the types package

## Build and Development Commands

### Run the API
```bash
# Development mode
cd src/Api/ProductApi
dotnet run

# API runs on http://localhost:5000
# Swagger UI at http://localhost:5000/swagger
# OpenAPI spec at http://localhost:5000/swagger/v1/swagger.json

# Release build
dotnet build src/Api/ProductApi -c Release
dotnet publish src/Api/ProductApi -c Release --self-contained -r linux-x64
```

### Generate TypeScript Types Locally
```bash
# 1. Start the API (in one terminal)
cd src/Api/ProductApi
dotnet run

# 2. In another terminal, fetch spec and generate types
curl http://localhost:5000/swagger/v1/swagger.json -o swagger.json
npx nswag openapi2tsclient /input:swagger.json /output:src/TypesPackage/src/models.ts /template:fetch /generateClientInterfaces:true /generateClientClasses:false /typeStyle:interface /enumStyle:enum

# 3. Create package structure (package.json, tsconfig.json, index.ts)
# See .gitlab-ci.yml for exact structure

# 4. Build the TypeScript package
cd src/TypesPackage
npm install
npm run build  # Compiles to dist/
```

### Run Frontend Demo
```bash
cd src/Frontend
npm install
npm run dev    # Vite dev server on http://localhost:5173
npm run build  # Production build to dist/
```

### Publish NPM Package to GitLab
The package is automatically published by CI/CD. For manual publishing:
```bash
cd src/TypesPackage
echo "//gitlab/api/v4/projects/${CI_PROJECT_ID}/packages/npm/:_authToken=${CI_JOB_TOKEN}" > .npmrc
npm publish --registry="http://gitlab/api/v4/projects/${CI_PROJECT_ID}/packages/npm/"
```

## Architecture and Key Files

### Type Generation Flow (Automated in CI/CD)
1. **C# Models** (`/src/Api/ProductApi/Models/`) define the source of truth - no attributes required
2. **API Build** compiles the .NET application
3. **API Starts Temporarily** during CI to serve OpenAPI spec
4. **OpenAPI Spec** fetched from `/swagger/v1/swagger.json`
5. **NSwag CLI** converts OpenAPI spec to TypeScript interfaces
6. **Generated Files** (in CI-generated `/src/TypesPackage/`):
   - `src/models.ts`: TypeScript interfaces generated by NSwag from OpenAPI
   - `src/index.ts`: Package exports (generated by CI)
   - `package.json`: Package configuration (generated by CI)
   - `tsconfig.json`: TypeScript compiler config (generated by CI)
   - `dist/`: Compiled JavaScript and type definitions

### GitLab CI/CD Pipeline Stages
1. **build-api**:
   - Builds .NET API
   - Starts API temporarily in background
   - Fetches OpenAPI/Swagger spec
   - Runs NSwag to generate TypeScript from spec
   - Creates package.json, tsconfig.json, index.ts inline
   - Artifacts: API publish/ + generated TypesPackage/

2. **generate-types**:
   - Receives TypesPackage from build-api artifacts
   - Runs `npm install` and `npm run build`
   - Compiles TypeScript to JavaScript + .d.ts files
   - Artifacts: Complete TypesPackage with dist/

3. **publish-npm-package**:
   - Publishes to GitLab Package Registry
   - Only runs on main/master/tags
   - Uses internal Docker URL for registry
   - Auto-increments version with CI_PIPELINE_IID

4. **build-frontend**:
   - Builds frontend demo
   - Uses published types package

5. **deploy-api-artifact**:
   - Creates API deployment artifact

### Key Configuration Files
- `/src/Api/ProductApi/ProductApi.csproj`: NSwag.AspNetCore 14.6.1, Swashbuckle 6.5.0
- `/src/Api/ProductApi/Program.cs`: Swagger enabled in ALL environments (required for CI)
- `/.gitlab-ci.yml`: 4-stage automated pipeline - generates TypesPackage inline
- `/.gitignore`: Excludes `src/TypesPackage/` (generated artifact)
- `/src/Frontend/package.json`: Consumes `@mycompany/api-types` package

### Important: TypesPackage is NOT in Source Control
The `src/TypesPackage/` directory is **generated during CI/CD** and should NOT be committed:
- Added to `.gitignore`
- Created fresh on each pipeline run
- All files (package.json, tsconfig.json, models.ts, index.ts) generated by CI

## Working with the Types System

### Adding New Models
1. Create C# model in `/src/Api/ProductApi/Models/` - **no attributes needed**
2. Add corresponding controller endpoints (models must be exposed in API)
3. Commit and push - CI/CD automatically:
   - Fetches new OpenAPI spec
   - Regenerates TypeScript with new types
   - Publishes updated package

Example model (no attributes required):
```csharp
namespace ProductApi.Models;

/// <summary>
/// Product description (appears in OpenAPI/TypeScript)
/// </summary>
public class Product
{
    public int Id { get; set; }
    public string Name { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public DateTime CreatedAt { get; set; }
}
```

### Model Examples
Current models include:
- **Product**: Complex model with nested dimensions, tags, timestamps
- **Order**: Includes enums (OrderStatus, PaymentStatus), nested objects (ShippingAddress, PaymentInfo)

All models are plain C# POCOs - no special attributes needed. Types are generated from the OpenAPI specification.

### Using Generated Types in Frontend
```typescript
import type { Product } from '@mycompany/api-types';

// Full IntelliSense and type safety
const product: Product = {
  id: 1,
  name: 'Sample',
  price: 99.99,
  createdAt: new Date()
};
```

## Type Mappings

NSwag generates accurate TypeScript types from OpenAPI:
- C# `int`, `double`, `decimal` → TypeScript `number`
- C# `string` → TypeScript `string`
- C# `DateTime` → TypeScript `Date`
- C# `bool` → TypeScript `boolean`
- C# `List<T>`, `T[]` → TypeScript `T[]`
- C# `T?` (nullable) → TypeScript `T | null`
- C# enums → TypeScript enums
- Nested objects preserved with correct structure

All properties are marked optional (`?`) in generated interfaces.

## Important Notes

- **OpenAPI/Swagger approach**: Uses industry-standard OpenAPI specification
- **No model attributes required**: Plain C# POCOs work automatically
- **Swagger in all environments**: Must be enabled in Production for CI to fetch spec
- **GitLab-centric deployment**: Package publishes to GitLab Package Registry
- **Self-contained API builds**: API publishes as standalone Linux executable
- **Local package development**: Frontend uses `file:../TypesPackage` for local testing
- **No prepare script**: TypeScript package doesn't run build on install (avoids CI issues)
- **Internal Docker URLs**: CI/CD uses `http://gitlab/...` for registry access
- **TypesPackage not committed**: Entire package generated fresh on each build

## CI/CD Requirements

- **Runners**: Pipeline doesn't use restrictive tags - runs on any available Docker-capable runner
- **Internal networking**: Uses `http://gitlab/api/v4/...` instead of external URLs
- **Node.js in .NET image**: Installs Node/npm in build-api job for NSwag CLI
- **API must start**: API runs temporarily during build to serve OpenAPI spec
- **Port awareness**: API runs on port 8080 in Production, 5000 in Development

## Troubleshooting

### Types not generating
- Check Swagger is enabled in ALL environments in Program.cs
- Verify API starts successfully (check api.log in CI artifacts)
- Ensure OpenAPI spec is accessible: curl http://localhost:8080/swagger/v1/swagger.json
- Check NSwag CLI output in build-api job logs

### CI Pipeline fails
- **Swagger 404**: Enable Swagger in Production environment
- **No runners**: Remove any `tags:` restrictions from jobs
- **Connection refused**: Use `http://gitlab/...` internal URLs
- **API won't start**: Increase sleep time or check for errors in api.log

### Type mismatches
- Verify models are exposed through API controllers (not exposed = not in OpenAPI spec)
- Check OpenAPI spec includes all expected models
- Rebuild and republish: pipeline regenerates everything from scratch
